const express = require("express");
const db = require("../models/db");
const multer = require("multer");
const xlsx = require("xlsx");

const router = express.Router();

// Multer setup for file upload
const upload = multer({ dest: "uploads/" });

// Helper function to generate unique custom IDs for each student
async function generateCustomId(session) {
  const [result] = await db.query(
    "SELECT COUNT(*) AS count FROM student WHERE session = ?",
    [session]
  );
  const nextSerialNumber = result[0].count + 1;
  return `DMC/${session}/${nextSerialNumber}`;
}

// Route to generate a custom ID based on the session
router.get("/generateCustomId/:session", async (req, res) => {
  try {
    const { session } = req.params;
    const custom_id = await generateCustomId(session);
    res.status(200).json({ custom_id });
  } catch (error) {
    console.error("Error generating custom ID:", error.message);
    res.status(500).json({ error: "Failed to generate custom ID" });
  }
});

// Helper function to validate and map `class_name` and `course_name` to IDs
async function validateAndMapIds(students) {
  const errors = [];
  const mappedStudents = [];

  for (const [index, student] of students.entries()) {
    const { name, class_name, course_name, session } = student;

    // Validate and map class_name to class_id
    const [classResult] = await db.query(
      "SELECT id FROM class WHERE class_name = ?",
      [class_name]
    );
    if (classResult.length === 0) {
      errors.push(`Row ${index + 1}: Invalid class name '${class_name}'`);
      continue;
    }
    const class_id = classResult[0].id;

    // Validate and map course_name to course_id within the specified class
    const [courseResult] = await db.query(
      "SELECT id FROM course WHERE course_name = ? AND class_id = ?",
      [course_name, class_id]
    );
    if (courseResult.length === 0) {
      errors.push(
        `Row ${
          index + 1
        }: Invalid course name '${course_name}' for class '${class_name}'`
      );
      continue;
    }
    const course_id = courseResult[0].id;

    // Generate unique custom ID for each student
    const custom_id = await generateCustomId(session);

    // Add mapped and validated student data
    mappedStudents.push({ custom_id, name, class_id, course_id, session });
  }

  return { errors, mappedStudents };
}

// Register a student with custom ID
router.post("/create", async (req, res) => {
  try {
    const { name, class_id, course_id, session } = req.body;
    const custom_id = await generateCustomId(session);

    const query =
      "INSERT INTO student (custom_id, name, class_id, course_id, session) VALUES (?, ?, ?, ?, ?)";
    const [result] = await db.query(query, [
      custom_id,
      name,
      class_id,
      course_id,
      session,
    ]);

    res.status(200).json({
      message: "Student created successfully",
      studentId: result.insertId,
    });
  } catch (err) {
    console.error("Error creating student:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Bulk upload students from Excel file with custom IDs
router.post("/create/bulk", upload.single("file"), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: "No file uploaded" });
    }

    const workbook = xlsx.readFile(req.file.path);
    const sheetName = workbook.SheetNames[0];
    const sheet = workbook.Sheets[sheetName];
    const students = xlsx.utils.sheet_to_json(sheet);

    // Validate and map IDs based on `class_name` and `course_name`
    const { errors, mappedStudents } = await validateAndMapIds(students);

    if (errors.length > 0) {
      return res.status(400).json({ error: errors });
    }

    const query =
      "INSERT INTO student (custom_id, name, class_id, course_id, session) VALUES ?";
    const values = mappedStudents.map((student) => [
      student.custom_id,
      student.name,
      student.class_id,
      student.course_id,
      student.session,
    ]);

    const [result] = await db.query(query, [values]);
    res.status(200).json({
      message: "Students created successfully",
      studentCount: result.affectedRows,
    });
  } catch (err) {
    console.error("Error during bulk student creation:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Get all students
router.get("/", async (req, res) => {
  try {
    const query = "SELECT * FROM student";
    const [results] = await db.query(query);
    res.json(results);
  } catch (err) {
    console.error("Error fetching all students:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Get a single student by ID
router.get("/:id", async (req, res) => {
  try {
    const { id } = req.params;
    const query = "SELECT * FROM student WHERE id = ?";
    const [result] = await db.query(query, [id]);

    if (result.length === 0) {
      return res.status(404).json({ message: "Student not found" });
    }
    res.json(result[0]);
  } catch (err) {
    console.error("Error fetching student by ID:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Update a student by ID
router.put("/:id", async (req, res) => {
  try {
    const { id } = req.params;
    const { name, class_id, course_id, session } = req.body;
    const query =
      "UPDATE student SET name = ?, class_id = ?, course_id = ?, session = ? WHERE id = ?";
    const [result] = await db.query(query, [
      name,
      class_id,
      course_id,
      session,
      id,
    ]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Student not found" });
    }
    res.status(200).json({ message: "Student updated successfully" });
  } catch (err) {
    console.error("Error updating student:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Delete a student by ID, including associated marks records
router.delete("/:id", async (req, res) => {
  try {
    const { id } = req.params;

    // First, delete related records in the marks table
    await db.query("DELETE FROM marks WHERE student_id = ?", [id]);

    // Then, delete the student record
    const [result] = await db.query("DELETE FROM student WHERE id = ?", [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: "Student not found" });
    }

    res
      .status(200)
      .json({ message: "Student and associated records deleted successfully" });
  } catch (err) {
    console.error("Error deleting student:", err.message);
    res.status(500).json({ error: "Failed to delete student" });
  }
});

// Route to get the total number of students
router.get("/count/count", async (req, res) => {
  const query = `
    SELECT COUNT(*) AS total_students
    FROM student;
  `;

  try {
    const [results] = await db.query(query);
    const totalStudents = results[0].total_students;

    res.status(200).json({
      success: true,
      total_students: totalStudents,
    });
  } catch (error) {
    console.error("Error fetching student count:", error.message);
    res
      .status(500)
      .json({ success: false, error: "Failed to fetch student count." });
  }
});

// Search students by name or custom ID
router.get("/api/student", async (req, res) => {
  const search = req.query.search || "";
  const students = await db.query(
    `SELECT * FROM students WHERE name LIKE ? OR custom_id LIKE ?`,
    [`%${search}%`, `%${search}%`]
  );
  res.json(students);
});

// Monthly data route
router.get("/monthly", async (req, res) => {
  try {
    const query = `
      SELECT 
        MONTH(created_at) AS month, 
        COUNT(*) AS total 
      FROM student 
      WHERE YEAR(created_at) = YEAR(CURDATE()) 
      GROUP BY month 
      ORDER BY month;
    `;

    const [results] = await db.query(query);

    const monthlyData = [
      { name: "Jan", total: 0 },
      { name: "Feb", total: 0 },
      { name: "Mar", total: 0 },
      { name: "Apr", total: 0 },
      { name: "May", total: 0 },
      { name: "Jun", total: 0 },
      { name: "Jul", total: 0 },
      { name: "Aug", total: 0 },
      { name: "Sep", total: 0 },
      { name: "Oct", total: 0 },
      { name: "Nov", total: 0 },
      { name: "Dec", total: 0 },
    ];

    results.forEach((result) => {
      monthlyData[result.month - 1].total = result.total;
    });

    res.status(200).json(monthlyData);
  } catch (error) {
    console.error("Error fetching monthly students data:", error.message);
    res.status(500).json({ error: "Failed to fetch monthly students data" });
  }
});

module.exports = router;
