const express = require("express");
const router = express.Router();
const db = require("../models/db"); // Ensure this uses mysql2/promise

// Create a subject
router.post("/create", async (req, res) => {
  const { subject_name, course_id, class_id, session } = req.body;

  // Validate request data
  if (!subject_name || !course_id || !class_id || !session) {
    return res.status(400).json({
      error:
        "All fields (subject_name, course_id, class_id, session) are required.",
    });
  }

  const query =
    "INSERT INTO subject (subject_name, course_id, class_id, session) VALUES (?, ?, ?, ?)";

  try {
    const [result] = await db.query(query, [
      subject_name,
      course_id,
      class_id,
      session,
    ]);
    res.status(200).json({
      message: "Subject created successfully",
      subjectId: result.insertId,
    });
  } catch (err) {
    console.error("Error creating subject:", err.message);
    return res.status(500).json({ error: err.message });
  }
});

// Get all subjects
router.get("/", async (req, res) => {
  const query = "SELECT * FROM subject";

  try {
    const [results] = await db.query(query);
    res.json(results);
  } catch (err) {
    console.error("Error fetching subjects:", err.message);
    return res.status(500).json({ error: err.message });
  }
});

// Get a single subject by ID
router.get("/:id", async (req, res) => {
  const { id } = req.params;
  const query = "SELECT * FROM subject WHERE id = ?";

  try {
    const [result] = await db.query(query, [id]);
    if (result.length === 0) {
      return res.status(404).json({ message: "Subject not found" });
    }
    res.json(result[0]);
  } catch (err) {
    console.error("Error fetching subject by ID:", err.message);
    return res.status(500).json({ error: err.message });
  }
});

// Update a subject
router.put("/:id", async (req, res) => {
  const { id } = req.params;
  const { subject_name, course_id, class_id, session } = req.body;

  if (!subject_name || !course_id || !class_id || !session) {
    return res.status(400).json({
      error:
        "All fields (subject_name, course_id, class_id, session) are required.",
    });
  }

  const query = `
    UPDATE subject
    SET subject_name = ?, course_id = ?, class_id = ?, session = ?
    WHERE id = ?
  `;

  try {
    const [result] = await db.query(query, [
      subject_name,
      course_id,
      class_id,
      session,
      id,
    ]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Subject not found" });
    }

    res.status(200).json({ message: "Subject updated successfully" });
  } catch (err) {
    console.error("Error updating subject:", err.message);
    return res.status(500).json({ error: err.message });
  }
});

// Delete a subject
router.delete("/:id", async (req, res) => {
  const { id } = req.params;
  const query = "DELETE FROM subject WHERE id = ?";

  try {
    const [result] = await db.query(query, [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Subject not found" });
    }

    res.status(200).json({ message: "Subject deleted successfully" });
  } catch (err) {
    console.error("Error deleting subject:", err.message);
    return res.status(500).json({ error: err.message });
  }
});

// Get subject distribution data
router.get("/distribution", async (req, res) => {
  const query = `
    SELECT subject.subject_name AS name, COUNT(marks.student_id) AS value
    FROM subject
    LEFT JOIN marks ON marks.subject_id = subject.id
    GROUP BY subject.subject_name
  `;

  try {
    const [results] = await db.query(query);
    res.json(results);
  } catch (err) {
    console.error("Error fetching subject distribution data:", err.message);
    res
      .status(500)
      .json({ error: "Failed to fetch subject distribution data" });
  }
});

module.exports = router;
